package de.jaggl.utils.common.helpers;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

public class ReflectionHelper
{

    public static void doWithMethods(Class<?> clazz, MethodCallback mc)
        throws IllegalArgumentException
    {
        Method[] methods = clazz.getDeclaredMethods();
        for (Method method : methods)
        {
            try
            {
                mc.doWith(method);
            }
            catch (IllegalAccessException ex)
            {
                throw new IllegalStateException(
                    "Shouldn't be illegal to access method '"
                        + method.getName()
                        + "': " + ex);
            }
        }
        if (clazz.getSuperclass() != null)
        {
            doWithMethods(clazz.getSuperclass(), mc);
        }
        else if (clazz.isInterface())
        {
            for (Class<?> superIfc : clazz.getInterfaces())
            {
                doWithMethods(superIfc, mc);
            }
        }
    }

    public static Method findMethod(Class<?> type, String name,
        Class<?>... parameterTypes)
    {
        try
        {
            return type.getDeclaredMethod(name, parameterTypes);
        }
        catch (Exception e)
        {
            return null;
        }
    }

    public static boolean parametersMatches(Method method, Object... arg)
    {
        if (method.getParameterTypes().length != arg.length)
        {
            return false;
        }
        int i = 0;
        for (Class<?> type : method.getParameterTypes())
        {
            if (!type.isAssignableFrom(arg[i++].getClass()))
            {
                return false;
            }
        }
        return true;
    }

    @SafeVarargs
    public static <E> boolean isOneOfType(E object, Class<? extends E>... types)
    {
        for (Class<? extends E> type : types)
        {
            if (type.isAssignableFrom(object.getClass()))
            {
                return true;
            }
        }
        return false;
    }

    public static boolean hasParameters(Method method)
    {
        return method.getParameterTypes().length > 0;
    }

    public static Object invoke(Method method, Object target, Object... args)
        throws IllegalAccessException, IllegalArgumentException
    {
        try
        {
            method.setAccessible(true);
            return method.invoke(target, args);
        }
        catch (InvocationTargetException e)
        {
            throw new RuntimeException(e);
        }
    }

    public static void setFieldData(Object object, String fieldname, Object data)
    {
        try
        {
            Field field = getField(object, fieldname);
            field.setAccessible(true);
            field.set(object, data);
        }
        catch (Exception e)
        {
            throw new RuntimeException(e);
        }
    }

    public static Object getFieldData(Object object, String fieldname)
    {
        try
        {
            Field field = getField(object, fieldname);
            field.setAccessible(true);
            return field.get(object);
        }
        catch (Exception e)
        {
            throw new RuntimeException(e);
        }
    }

    public static Field getField(Object object, String fieldName)
        throws NoSuchFieldException
    {
        Field field = collectFields(object).get(fieldName);
        if (field == null)
        {
            throw new NoSuchFieldException(fieldName);
        }
        return field;
    }

    public static Map<String, Field> collectFields(Object object)
    {
        Map<String, Field> allFields = new HashMap<String, Field>();

        Class<?> type = object.getClass();
        while (type != null)
        {
            Field[] fields = type.getDeclaredFields();
            for (Field field : fields)
            {
                if (!allFields.containsKey(field.getName()))
                {
                    allFields.put(field.getName(), field);
                }
            }
            type = type.getSuperclass();
        }
        return allFields;
    }

    public interface MethodCallback
    {

        void doWith(Method method) throws IllegalAccessException,
            IllegalArgumentException;

    }

}
