package de.jaggl.utils.common.helpers;

import static java.util.Arrays.asList;
import static java.util.Collections.singleton;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.swing.ListModel;

public class ListHelper
{

    public static <S, T> List<T> convertAll(List<S> sourceItems,
        Converter<S, T> converter)
    {
        List<T> targetList = new ArrayList<T>();
        for (S sourceItem : sourceItems)
        {
            targetList.add(converter.convert(sourceItem));
        }
        return targetList;
    }

    public static <T> Collection<T> removeNullValues(Collection<T> items)
    {
        Collection<T> collection = new ArrayList<T>(items);
        collection.removeAll(singleton(null));
        return collection;
    }

    public static <T> Collection<Collection<T>> divideInParts(
        Collection<T> items, int maxItems)
    {
        if (maxItems <= 0)
        {
            throw new ArithmeticException("maxItems must be greater than 0");
        }
        List<Collection<T>> parts = new ArrayList<Collection<T>>();
        if (items == null)
        {
            return parts;
        }
        int count = items.size() / maxItems;
        count = (items.size() % maxItems != 0) ? count + 1 : count;
        for (int i = 0; i < count; i++)
        {
            int fromIndex = i * maxItems;
            int toIndex = (i + 1) * maxItems;
            toIndex = (toIndex > items.size()) ? items.size() : toIndex;
            List<T> part = new ArrayList<T>(items).subList(fromIndex, toIndex);
            parts.add(part);
        }
        return parts;
    }

    public static <T> Collection<T> merge(Collection<Collection<T>> lists)
    {
        List<T> resultList = new ArrayList<T>();
        if (lists == null)
        {
            return resultList;
        }
        for (Collection<T> list : lists)
        {
            if (list != null)
            {
                resultList.addAll(list);
            }
        }
        return resultList;
    }

    @SafeVarargs
    public static <T> boolean inList(T value, T... list)
    {
        if (value == null)
        {
            return false;
        }
        for (T item : list)
        {
            if (item != null)
            {
                if (value.equals(item))
                {
                    return true;
                }
            }
        }
        return false;
    }

    public static <E> String implodeList(String glue, Collection<E> list)
    {
        if (list == null)
        {
            return "";
        }
        StringBuilder builder = new StringBuilder();
        int i = 0;
        for (Object value : list)
        {
            builder.append(value);
            if (++i < list.size())
            {
                builder.append(glue);
            }
        }
        return builder.toString();
    }

    public static String implodeArgs(String glue, Object... args)
    {
        return implodeList(glue, asList(args));
    }

    public static String implodeArray(String glue, Object[] args)
    {
        return implodeList(glue, asList(args));
    }

    public static <T> Collection<T> getElements(ListModel<T> listModel)
    {
        List<T> elements = new ArrayList<T>();
        for (int i = 0; i < listModel.getSize(); i++)
        {
            elements.add(listModel.getElementAt(i));
        }
        return elements;
    }

    public static <E> Collection<E> multiplyEntries(Collection<E> list,
        int factor)
    {
        if (list == null)
        {
            return null;
        }
        List<E> result = new ArrayList<E>();
        for (E entry : list)
        {
            for (int i = 0; i < factor; i++)
            {
                result.add(entry);
            }
        }
        return result;
    }

    public static <E> E[] addFirstToArray(E[] array, E value)
    {
        List<E> list = new ArrayList<E>();
        list.add(value);
        list.addAll(asList(array));
        return list.toArray(array);
    }

    @SafeVarargs
    public static <T> T[] addToArray(T[] args, T... values)
    {
        List<T> list = new ArrayList<T>();
        list.addAll(asList(args));
        for (T value : values)
        {
            list.add(value);
        }
        return list.toArray(args);
    }

    @SafeVarargs
    public static <T> T[] toArray(T... values)
    {
        return values;
    }

    public interface Converter<S, T>
    {

        T convert(S source);

    }

}
